C      *****************************************************************
C      * Subroutine CalcActCoeff                                       *
C      * Calculates the activity coefficient for each species.         *
C      *                                                               *
C      * Variables:                                                    *
C      * ActCorr   -Input - A vector, of size NSpecies, of flags to    *
C      *                    indicate how activity corrections will be  *
C      *                    calculated.                                *
C      *                    1 = None (concentration = activity),       *
C      *                    2 = Davies,                                *
C      *                    3 = Debye-Huckle (not included but easy to *
C      *                        add if desired),                       *
C      *                    4 = Vanselow exchange,                     *
C      *                    5 = Gaines-Thomas exchange.                *
C      *                    Unitless.                                  *
C      *                    (Common block VModel, file VModel.f)       *
C      * Dav       -Local - Flag indicating Davies activity correction *
C      *                    choice.  Unitless.                         *
C      * Davies    -Fcn   - Returns Davies activity correction for a   *
C      *                    given ionic charge and ionic strength.     *
C      *                    Unitless.                                  *
C      * DebHuck   -Local - Flag indicating Debye-Huckle activity      *
C      *                    correction choice.  Unitless.              *
C      * Denom     -Local - Sum of species, from one to NList, to be   *
C      *                    used as denominator of Vanselow or         *
C      *                    Gaines-Thomas exchange expressions.  Units *
C      *                    are either mol or mol-equivalents.         *
C      * GainThom  -Local - Flag indicating Gaines-Thomas activity     *
C      *                    correction choice.  Unitless.              *
C      * Gamma     -Output- A vector, of size NSpecies, of activity    *
C      *                    coefficients.  Units are L/mol.            *
C      * GoAhead   -Output- Indicates whether any errors were found.   *
C      *                    (GoAhead=1, no errors found; GoAhead=0,    *
C      *                    errors found.)  Unitless.                  *
C      * IonicStr  -Input - Ionic strength.  Units are mol/L.  (These  *
C      *                    units do not result from the calculation   *
C      *                    but are defined by usage (Morel 1983).)    *
C      * J         -Local - Loop index.  Unitless.                     *
C      * K         -Local - Loop index.  Unitless.                     *
C      * LinkComp  -Input - A vector, of size NSpecies, that specifies *
C      *                    the component with which a surface-bound   *
C      *                    species is linked.  Unitless.              *
C      *                    (Common block VReac, file VModel.f)        *
C      * LinkList  -Input - A matrix, of size NComp by NSpecies + 1,   *
C      *                    that lists the index numbers of the species*
C      *                    that will be used in the exchange reaction.*
C      *                    The zeroeth position for each row contains *
C      *                    the number of items in that row.  Unitless.*
C      *                    (Common block VReac, file VModel.f)        *
C      * LinkPos   -Input - A vector, of size NComp, of the indices for*
C      *                    the controlling component for a given      *
C      *                    position in the list.  Unitless.           *
C      *                    (Common block VReac, file VModel.f)        *
C      * LNum      -Input - The current layer number.  Unitless.       *
C      * N         -Local - Loop index.  Unitless.                     *
C      * NList     -Input - The number of items in the linked list,    *
C      *                    LinkList.  Unitless.                       *
C      *                    (Common block VReac, file VModel.f)        *
C      * NComp     -Input - The number of components in the system.    *
C      *                    Unitless.                                  *
C      *                    (Common block VModel, file VModel.f)       *
C      * NoChoice  -Local - Flag indicating no choice for activity     *
C      *                    correction choice.  Unitless.              *
C      * NSpeciesSize                                                  *
C      *           -Input - The max number of species, used to size    *
C      *                    arrays.  Unitless.                         *
C      *                    (File ArraySizes.Inc)                      *
C      * NSpecies  -Input - The number of species in the system (i.e.  *
C      *                    the number of components plus the number   *
C      *                    of dependent species).  Unitless.          *
C      *                    (Common block VModel, file VModel.f)       *
C      * S         -Local - Loop index.  Unitless.                     *
C      * SpConc    -Input - A matrix, of size NSpecies by NLayers, of  *
C      *                    species concentrations, one set for each   *
C      *                    layer. Units are mol/L.                    *
C      *                    (Common block VModel, file VModel.f)       *
C      * V         -Input - A vector, of size NSpecies, of the ionic   *
C      *                    charge for each species.  Unitless.        *
C      * Vanse     -Local - Flag indicating Vanselow activity for      *
C      *                    activity correction choice.  Unitless.     *
C      * Z         -Input - A vector, of size NComp, of unspeciated    *
C      *                    charge for each component.  Units are      *
C      *                    equivalent charge.                         *
C      *                    (Common block VModel, file VModel.f)       *
C      *****************************************************************
       SUBROUTINE CALCACTCOEFF(LNUM, IONICSTR, GOAHEAD, GAMMA)
       IMPLICIT NONE
       INCLUDE 'VMODEL.INC'
       REAL*8 DAVIES
       REAL*8 DENOM(25)  !Just has to be bigger than a possible NList
       REAL*8 GAMMA(NSPECIESSIZE), IONICSTR

       INTEGER DAV, DEBHUCK, GAINTHOM
       INTEGER GOAHEAD, ITEMP, J, K
       INTEGER LNUM, N, NOCHOICE, S, VANSE

       PARAMETER (NOCHOICE = 1, DAV = 2, DEBHUCK = 3, VANSE = 4)
       PARAMETER (GAINTHOM = 5)

C      *------------------------------------*
C      * Indicate that there are no errors. *
C      *------------------------------------*
       GOAHEAD = 1

C      *--------------------*
C      * Clear denom array. *
C      *--------------------*
       DO 10 N=1, NLIST
          DENOM(N) = 0.0
  10   CONTINUE

C      *------------------------------*
C      * Do preliminary calculations. *
C      *------------------------------*
       DO 300 N=1, NLIST
          ITEMP = ACTCORR(LINKLIST(N, 1))
C         *-------------------------------------------------------------*
C         * If this item indicates Vanselow exchange then calculate the *
C         * sum of all Vanselow exchange species in this list, in moles.*
C         *-------------------------------------------------------------*
          IF (ITEMP.EQ.VANSE) THEN
             DO 100 J=1, LINKLIST(N, 0)
                DENOM(N) = DENOM(N) + SPCONC(LINKLIST(N, J),LNUM)
  100        CONTINUE
C         *--------------------------------------------------------------*
C         * If this item indicates Gaines-Thomas exchange then calculate *
C         * the sum of all Gaines-Thomas exchange species in this list,  *
C         * in equivalents.                                              *
C         *--------------------------------------------------------------*
          ELSEIF (ITEMP.EQ.GAINTHOM) THEN
             DO 200, J=1, LINKLIST(N, 0)
                DENOM(N) = DENOM(N) + SPCONC(LINKLIST(N, J),LNUM) * 
     >                     Z(LINKLIST(N,J))
  200        CONTINUE
          ELSE
C            *--------------------------------------------------------*
C            * This is trouble and indicates that an error was made   *
C            * when the linked list was set up.                       *
C            * An error message and program termination should result.*
C            *--------------------------------------------------------*
             GOAHEAD = 0
             WRITE(6,*) 'Error in routine CalcActCoeff - linked '
             WRITE(6,*) 'list was set up incorrectly'
             WRITE(YASOUT,*) 'Error in routine CalcActCoeff - linked '
             WRITE(YASOUT,*) 'list was set up incorrectly'
             GOTO 9999
          END IF

C         *-----------------------------------------------------------*
C         * If zero activity corrections result (theoretically        *
C         * impossible) they may be due to weird species              *
C         * concentrations that happen in early iterations toward the *
C         * solution.  I can't think of how to test for that case in  *
C         * particular, but if it happens for any other reason it     *
C         * could mean trouble.  The solution in the first case is to * 
C         * set DENOM to 1 and expect it to go away.  This solution   *
C         * would mask the presence of other reasons (i.e., bugs) and *
C         * so it is a potentially worrisome thing to code in.        *
C         *-----------------------------------------------------------*
          IF (DENOM(N).EQ.0) THEN 
             DENOM(N) = 1.0
             WRITE(6,*) 
     >       'Theoretically impossible happened in CalcActCoeff'
             WRITE(YASOUT,*) 
     >       'Theoretically impossible happened in CalcActCoeff'
          ENDIF
  300  CONTINUE

C      *-----------------------------------------------------------*
C      * All the preliminaries are done, now calculate the         *
C      * coefficients and store them in GAMMA.                     *
C      *-----------------------------------------------------------*
       DO 400 S=1, NSPECIES
          ITEMP = ACTCORR(S)
          IF (ITEMP.EQ.NOCHOICE) THEN
C            *-------------------------*
C            * No activity correction  *
C            *-------------------------*
             GAMMA(S) = 1
          ELSEIF (ITEMP.EQ.DAV) THEN
C            *-------------------------*
C            * Davies correction       *
C            *-------------------------*
             GAMMA(S) = DAVIES(IONICSTR, Z(S))
          ELSEIF (ITEMP.EQ.VANSE) THEN
C            *-------------------------*
C            * Vanselow exchange       *
C            *-------------------------*
             GAMMA(S) = 1 / DENOM(LINKPOS(LINKCOMP(S)))
          ELSEIF (ITEMP.EQ.GAINTHOM) THEN
C            *-------------------------*
C            * Gaines-Thomas exchange  *
C            *-------------------------*
             GAMMA(S) = Z(S) / DENOM(LINKPOS(LINKCOMP(S)))
          ELSE
C            *--------------------------------------------------------*
C            * Once again, this should never happen, but if it does   *
C            * it means there is trouble somewhere (probably in the   *
C            * interface).                                            *
C            *--------------------------------------------------------*
             GOAHEAD = 0
             WRITE(6,*) 'Error in routine CalcActCoeff - linked '
             WRITE(6,*) 'list was set up incorrectly.  Each list'
             WRITE(6,*) 'must indicate either Vanselow or '
             WRITE(6,*) 'Gaines-Thomas.'
             WRITE(YASOUT,*) 'Error in routine CalcActCoeff - linked '
             WRITE(YASOUT,*) 'list was set up incorrectly.  Each list'
             WRITE(YASOUT,*) 'must indicate either Vanselow or '
             WRITE(YASOUT,*) 'Gaines-Thomas.'
             GOTO 9999
          END IF
  400   CONTINUE

C      *--------------*
C      * Escape hatch *
C      *--------------*
 9999  CONTINUE

       RETURN
       END
C      ****************************************************************
C      *               END OF SUBROUTINE CALCACTCOEFF                 *
C      ****************************************************************



